# ----------------------------------------------------------------------
# NEC V60/V70/V80 processor module
# Copyright (C) 2019 Alex "trap15" Marshall <trap15@raidenii.net>
#
# TODO:
#  - Stack tracking
#  - Shift instruction pseudo-ops for sanity?
#  - Character instructions
#  - Bit string instructions
#  - Bit field instructions
#  - Remove (most) exceptions
#  - ldpr/stpr register names (noted in source)
#  - Register indirect should be a displacement type, so it can be used for structs
#  - MOV with uneven sizes marks addresses as wrong type
#  - Direct address modes are literally in the syntax `/myaddr` <-- other addresses are PC-rel
#  - `pushm   #0m<r0-r29>` WutFace
#

import sys
import copy
from ida_bytes import *
from ida_ua import *
from ida_idp import *
from ida_auto import *
from ida_nalt import *
import ida_frame
from ida_funcs import *
from ida_lines import *
from ida_problems import *
from ida_offset import *
from ida_segment import *
from ida_name import *
from ida_netnode import *
from ida_xref import *

# extract bitfield occupying bits high..low from val (inclusive, start from 0)
def BITS(val, high, low):
    return (val>>low)&((1<<(high-low+1))-1)

# extract one bit
def BIT(val, bit):
    return (val>>bit) & 1

# sign extend b low bits in x
# from "Bit Twiddling Hacks"
def SIGNEXT(x, b):
    m = 1 << (b - 1)
    x = x & ((1 << b) - 1)
    return (x ^ m) - m

def ZEROEXT(x, b):
    return x & ((1 << b) - 1)

ImmediatesAreSignExtended = False
VerboseFailure = False

def IMMEXT(x, b):
    if ImmediatesAreSignExtended:
        return SIGNEXT(x, b)
    else:
        return ZEROEXT(x, b)
def DISPEXT(x, b):
    return SIGNEXT(x, b)

# addressing mode field in the opcode
AddrMode_Register                       = 0     # Rn
AddrMode_RegisterIndirect               = 1     # [Rn]                  @[Rn]
AddrMode_Autoincrement                  = 2     # [Rn+]                 @[Rn+]
AddrMode_Autodecrement                  = 3     # [-Rn]                 @[-Rn]
AddrMode_Displacement                   = 4     # disp[Rn]              offset@[Rn]
AddrMode_DisplacementIndirect           = 5     # [disp[Rn]]            @[disp[Rn]]
AddrMode_DoubleDisplacement             = 6     # disp1[disp2[Rn]]      offset@[disp[Rn]]
AddrMode_DirectAddress                  = 7     # /addr                 @/addr
AddrMode_DirectAddressDeferred          = 8     # [/addr]               @[/addr]
AddrMode_Immediate                      = 9     # #value
AddrModeFlag_Indexed                    = 0x80  #
AddrModeFlagMask                        = 0x80

InsnAux_Postfix_Mask    =   0x3F
InsnAux_Postfix_None    =   0x00
InsnAux_Postfix_B       =   0x01
InsnAux_Postfix_H       =   0x02
InsnAux_Postfix_W       =   0x03
InsnAux_Postfix_D       =   0x04
InsnAux_Postfix_BH      =   0x05
InsnAux_Postfix_BW      =   0x06
InsnAux_Postfix_HB      =   0x07
InsnAux_Postfix_HW      =   0x08
InsnAux_Postfix_WB      =   0x09
InsnAux_Postfix_WH      =   0x0A

InsnFmt_I       =   0
InsnFmt_II      =   1
InsnFmt_I_II    =   2
InsnFmt_III     =   3
InsnFmt_IV      =   4
InsnFmt_V       =   5
InsnFmt_VI      =   6
InsnFmt_VIIa    =   7
InsnFmt_VIIb    =   8
InsnFmt_VIIc    =   9
InsnFmt_VIId    =   10

# operand data length value
DataLen_Byte    = 0 # 8-bit byte
DataLen_Hword   = 1 # 16-bit half-word
DataLen_Word    = 2 # 32-bit word
DataLen_Dword   = 3 # 64-bit double-word
DataLen_Offset  = 4 # Offset (no size)

dtype_size_bits = {dt_byte: 8,
                   dt_word: 16,
                   dt_dword: 32,
                   dt_qword: 64,
                  }
sizename_datalen_mapping = {'B': DataLen_Byte,
                            'H': DataLen_Hword,
                            'W': DataLen_Word,
                            'D': DataLen_Dword,
                            'O': DataLen_Offset,
                           }
datalen_dtype_mapping = {DataLen_Byte: dt_byte,
                         DataLen_Hword: dt_word,
                         DataLen_Word: dt_dword,
                         DataLen_Dword: dt_qword,
                         DataLen_Offset: dt_void,
                        }

# check if operand is immediate value val
def is_imm_op(op, val):
    if op.type == o_imm:
        # workaround for difference between Python and native numbers
        op2 = op_t()
        op2.value = val
        return op.value == op2.value
    return False

# are operands equal?
def same_op(op1, op2):
    return op1.type  == op2.type  and \
           op1.reg   == op2.reg   and \
           op1.value == op2.value and \
           op1.addr  == op2.addr  and \
           op1.flags == op2.flags and \
           op1.specval == op2.specval and \
           op1.dtype == op2.dtype

# ----------------------------------------------------------------------
class necv60_processor_t(processor_t):
    """
    Processor module classes must derive from processor_t
    """

    # IDP id ( Numbers above 0x8000 are reserved for the third-party modules)
    id = 0x8706 # uPD706** series

    # Processor features
    flag = PR_SEGS | PR_DEFSEG32 | PR_USE32 | PRN_HEX | PR_RNAMESOK | PR_NO_SEGMOVE

    # Number of bits in a byte for code segments (usually 8)
    # IDA supports values up to 32 bits
    cnbits = 8

    # Number of bits in a byte for non-code segments (usually 8)
    # IDA supports values up to 32 bits
    dnbits = 8

    # short processor names
    # Each name should be shorter than 9 characters
    psnames = ['necv60']

    # long processor names
    # No restriction on name lengthes.
    plnames = ['NEC V60/V70/V80']

    # size of a segment register in bytes
    segreg_size = 0

    # Array of typical code start sequences (optional)
    # codestart = ['\x0B\x12']  # 120B: push R11

    # Array of 'return' instruction opcodes (optional)
    # retcodes = ['\x30\x41']   # 4130: ret (mov.w @SP+, PC)

    decoder_table = [
        # data transfer instructions
        {'name': 'mov',     'valuemask': '00SS1ss1',    'format': InsnFmt_I_II, 'feature': ['DWORD_MUST_BOTH']},
        {'name': 'movs',    'valuemask': '00SS1ss0',    'format': InsnFmt_I_II},
        {'name': 'xch',     'valuemask': '01000ss1',    'format': InsnFmt_I_II},
        {'name': 'movea',   'valuemask': '01000ss0',    'format': InsnFmt_I_II, 'size': 'O'},
        {'name': 'rvbyt',   'valuemask': '00101100',    'format': InsnFmt_I_II, 'size': 'W'},
        {'name': 'rvbit',   'valuemask': '00001000',    'format': InsnFmt_I_II, 'size': 'B'},

        # integer arithmetic instructions
        {'name': 'add',     'valuemask': '10000ss0',    'format': InsnFmt_I_II},
        {'name': 'addc',    'valuemask': '10010ss0',    'format': InsnFmt_I_II},
        {'name': 'sub',     'valuemask': '10101ss0',    'format': InsnFmt_I_II},
        {'name': 'subc',    'valuemask': '10011ss0',    'format': InsnFmt_I_II},
        {'name': 'mul',     'valuemask': '10000ss1',    'format': InsnFmt_I_II},
        {'name': 'mulu',    'valuemask': '10010ss1',    'format': InsnFmt_I_II},
        {'name': 'mulx',    'valuemask': '10000110',    'format': InsnFmt_I_II, 'size': ['W', 'D']},
        {'name': 'mulux',   'valuemask': '10010110',    'format': InsnFmt_I_II, 'size': ['W', 'D']},
        {'name': 'div',     'valuemask': '10100ss1',    'format': InsnFmt_I_II},
        {'name': 'divu',    'valuemask': '10110ss1',    'format': InsnFmt_I_II},
        {'name': 'divx',    'valuemask': '10100110',    'format': InsnFmt_I_II, 'size': ['W', 'D']},
        {'name': 'divux',   'valuemask': '10110110',    'format': InsnFmt_I_II, 'size': ['W', 'D']},
        {'name': 'rem',     'valuemask': '01010ss0',    'format': InsnFmt_I_II},
        {'name': 'remu',    'valuemask': '01010ss1',    'format': InsnFmt_I_II},
        {'name': 'inc',     'valuemask': '11011ssm',    'format': InsnFmt_III},
        {'name': 'dec',     'valuemask': '11010ssm',    'format': InsnFmt_III},
        {'name': 'neg',     'valuemask': '00111ss1',    'format': InsnFmt_I_II},
        {'name': 'cmp',     'valuemask': '10111ss0',    'format': InsnFmt_I_II},
        {'name': 'test',    'valuemask': '11110ssm',    'format': InsnFmt_III},

        # logical instructions
        {'name': 'and',     'valuemask': '10100ss0',    'format': InsnFmt_I_II},
        {'name': 'or',      'valuemask': '10001ss0',    'format': InsnFmt_I_II},
        {'name': 'xor',     'valuemask': '10110ss0',    'format': InsnFmt_I_II},
        {'name': 'not',     'valuemask': '00111ss0',    'format': InsnFmt_I_II},

        # shift/rotate instructions
        {'name': 'sha',     'valuemask': '10111ss1',    'format': InsnFmt_I_II, 'feature': ['IS_SHIFT']},
        {'name': 'shl',     'valuemask': '10101ss1',    'format': InsnFmt_I_II, 'feature': ['IS_SHIFT']},
        {'name': 'rot',     'valuemask': '10001ss1',    'format': InsnFmt_I_II, 'feature': ['IS_SHIFT']},
        {'name': 'rotc',    'valuemask': '10011ss1',    'format': InsnFmt_I_II, 'feature': ['IS_SHIFT']},

        # floating point instructions
        {'name': 'movf',    'valuemask': '00001000',    'format': InsnFmt_II,   'prefix': 'FPU'},
        {'name': 'addf',    'valuemask': '00011000',    'format': InsnFmt_II,   'prefix': 'FPU'},
        {'name': 'subf',    'valuemask': '00011001',    'format': InsnFmt_II,   'prefix': 'FPU'},
        {'name': 'mulf',    'valuemask': '00011010',    'format': InsnFmt_II,   'prefix': 'FPU'},
        {'name': 'divf',    'valuemask': '00011011',    'format': InsnFmt_II,   'prefix': 'FPU'},
        {'name': 'cmpf',    'valuemask': '00000000',    'format': InsnFmt_II,   'prefix': 'FPU'},
        {'name': 'negf',    'valuemask': '00001001',    'format': InsnFmt_II,   'prefix': 'FPU'},
        {'name': 'absf',    'valuemask': '00001010',    'format': InsnFmt_II,   'prefix': 'FPU'},
        {'name': 'sclf',    'valuemask': '00010000',    'format': InsnFmt_II,   'prefix': 'FPU'},
        {'name': 'cvt.ws',  'valuemask': '00000000',    'format': InsnFmt_II,   'size': ['W', 'W'], 'prefix': 'CVT'},
        {'name': 'cvt.sw',  'valuemask': '00000001',    'format': InsnFmt_II,   'size': ['W', 'W'], 'prefix': 'CVT'},
        {'name': 'cvt.ls',  'valuemask': '00001000',    'format': InsnFmt_II,   'size': ['D', 'W'], 'prefix': 'CVT'},
        {'name': 'cvt.lw',  'valuemask': '00001001',    'format': InsnFmt_II,   'size': ['D', 'W'], 'prefix': 'CVT'},
        {'name': 'cvt.sl',  'valuemask': '00010000',    'format': InsnFmt_II,   'size': ['W', 'D'], 'prefix': 'CVT'},
        {'name': 'cvt.wl',  'valuemask': '00010001',    'format': InsnFmt_II,   'size': ['D', 'W'], 'prefix': 'CVT'},
        {'name': 'trapfl',  'valuemask': '11001011',    'format': InsnFmt_V},

        # decimal arithmetic instructions
        {'name': 'adddc',   'valuemask': '---00000',    'format': InsnFmt_VIId, 'size': 'B', 'prefix': 'BCD'},
        {'name': 'subdc',   'valuemask': '---00001',    'format': InsnFmt_VIId, 'size': 'B', 'prefix': 'BCD'},
        {'name': 'subrdc',  'valuemask': '---00010',    'format': InsnFmt_VIId, 'size': 'B', 'prefix': 'BCD'},
        {'name': 'cvtd.pz', 'valuemask': '---10000',    'format': InsnFmt_VIId, 'size': ['B', 'H'], 'prefix': 'BCD'},
        {'name': 'cvtd.zp', 'valuemask': '---11000',    'format': InsnFmt_VIId, 'size': ['H', 'B'], 'prefix': 'BCD'},

        # bit manipulation instructions
        {'name': 'test1',   'valuemask': '10000111',    'format': InsnFmt_I_II, 'size': 'W'},
        {'name': 'set1',    'valuemask': '10010111',    'format': InsnFmt_I_II, 'size': 'W'},
        {'name': 'clr1',    'valuemask': '10100111',    'format': InsnFmt_I_II, 'size': 'W'},
        {'name': 'not1',    'valuemask': '10110111',    'format': InsnFmt_I_II, 'size': 'W'},

        # bit field instructions
        # extbfs
        # extbfz
        # extbfl
        # insbfr
        # insbfl
        # cmpbfs
        # cmpbfz
        # cmpbfl

        # bit string instructions
        # movbs
        # notbs
        # andbs
        # andnbs
        # orbs
        # ornbs
        # xorbs
        # xornbs
        # sch0bs
        # sch1bs

        # character manipulation instructions
        # movc
        # movcf
        # movcs
        # cmpc
        # cmpcf
        # cmpcs
        # schc
        # skpc

        # stack manipulation instructions
        {'name': 'push',    'valuemask': '1110111m',    'format': InsnFmt_III, 'size': 'W'},
        {'name': 'pop',     'valuemask': '1110011m',    'format': InsnFmt_III, 'size': 'W'},
        {'name': 'pushm',   'valuemask': '1110110m',    'format': InsnFmt_III, 'size': 'W'},
        {'name': 'popm',    'valuemask': '1110010m',    'format': InsnFmt_III, 'size': 'W'},
        {'name': 'prepare', 'valuemask': '1101111m',    'format': InsnFmt_III, 'size': 'W'},
        {'name': 'dispose', 'valuemask': '11001100',    'format': InsnFmt_V},

        # control transfer instructions
        {'name': 'bv',      'valuemask': '011s0000',    'format': InsnFmt_IV},
        {'name': 'bnv',     'valuemask': '011s0001',    'format': InsnFmt_IV},
        {'name': 'bc',      'valuemask': '011s0010',    'format': InsnFmt_IV},
        {'name': 'bnc',     'valuemask': '011s0011',    'format': InsnFmt_IV},
        {'name': 'bz',      'valuemask': '011s0100',    'format': InsnFmt_IV},
        {'name': 'bnz',     'valuemask': '011s0101',    'format': InsnFmt_IV},
        {'name': 'bnh',     'valuemask': '011s0110',    'format': InsnFmt_IV},
        {'name': 'bh',      'valuemask': '011s0111',    'format': InsnFmt_IV},
        {'name': 'bn',      'valuemask': '011s1000',    'format': InsnFmt_IV},
        {'name': 'bp',      'valuemask': '011s1001',    'format': InsnFmt_IV},
        {'name': 'br',      'valuemask': '011s1010',    'format': InsnFmt_IV},
        {'name': 'blt',     'valuemask': '011s1100',    'format': InsnFmt_IV},
        {'name': 'bge',     'valuemask': '011s1101',    'format': InsnFmt_IV},
        {'name': 'ble',     'valuemask': '011s1110',    'format': InsnFmt_IV},
        {'name': 'bgt',     'valuemask': '011s1111',    'format': InsnFmt_IV},

        {'name': 'dbv',     'valuemask': '000-----',    'format': InsnFmt_VI, 'prefix': 'DBcc'},
        {'name': 'dbc',     'valuemask': '001-----',    'format': InsnFmt_VI, 'prefix': 'DBcc'},
        {'name': 'dbz',     'valuemask': '010-----',    'format': InsnFmt_VI, 'prefix': 'DBcc'},
        {'name': 'dbnh',    'valuemask': '011-----',    'format': InsnFmt_VI, 'prefix': 'DBcc'},
        {'name': 'dbn',     'valuemask': '100-----',    'format': InsnFmt_VI, 'prefix': 'DBcc'},
        {'name': 'dbr',     'valuemask': '101-----',    'format': InsnFmt_VI, 'prefix': 'DBcc'},
        {'name': 'dblt',    'valuemask': '110-----',    'format': InsnFmt_VI, 'prefix': 'DBcc'},
        {'name': 'dble',    'valuemask': '111-----',    'format': InsnFmt_VI, 'prefix': 'DBcc'},
        {'name': 'dbnv',    'valuemask': '000-----',    'format': InsnFmt_VI, 'prefix': 'DBNcc'},
        {'name': 'dbnc',    'valuemask': '001-----',    'format': InsnFmt_VI, 'prefix': 'DBNcc'},
        {'name': 'dbnz',    'valuemask': '010-----',    'format': InsnFmt_VI, 'prefix': 'DBNcc'},
        {'name': 'dbh',     'valuemask': '011-----',    'format': InsnFmt_VI, 'prefix': 'DBNcc'},
        {'name': 'dbp',     'valuemask': '100-----',    'format': InsnFmt_VI, 'prefix': 'DBNcc'},
        {'name': 'dbge',    'valuemask': '110-----',    'format': InsnFmt_VI, 'prefix': 'DBNcc'},
        {'name': 'dbgt',    'valuemask': '111-----',    'format': InsnFmt_VI, 'prefix': 'DBNcc'},

        {'name': 'tb',      'valuemask': '101-----',    'format': InsnFmt_VI,   'prefix': 'DBNcc'},
        {'name': 'jmp',     'valuemask': '1101011m',    'format': InsnFmt_III,  'size': 'B', 'feature': ['CODE_DST']},
        {'name': 'bsr',     'valuemask': '01001000',    'format': InsnFmt_IV,   'size': 'H', 'feature': ['CODE_DST']},
        {'name': 'jsr',     'valuemask': '1110100m',    'format': InsnFmt_III,  'size': 'B', 'feature': ['CODE_DST']},
        {'name': 'rsr',     'valuemask': '11001010',    'format': InsnFmt_V},
        {'name': 'call',    'valuemask': '01001001',    'format': InsnFmt_I_II, 'size': ['O', 'W'], 'feature': ['CODE_DST']},
        {'name': 'ret',     'valuemask': '1110001m',    'format': InsnFmt_III,  'size': 'W'},
        {'name': 'brk',     'valuemask': '11001000',    'format': InsnFmt_V},
        {'name': 'brkv',    'valuemask': '11001001',    'format': InsnFmt_V},
        {'name': 'trap',    'valuemask': '1111100m',    'format': InsnFmt_III,  'size': 'B'},
        {'name': 'retiu',   'valuemask': '1110101m',    'format': InsnFmt_III,  'size': 'H'},

        # miscellaneous instructions
        {'name': 'nop',     'valuemask': '11001101',    'format': InsnFmt_V},
        {'name': 'getpsw',  'valuemask': '1111011m',    'format': InsnFmt_III,  'size': 'W'},
        {'name': 'updpsw.h','valuemask': '01001010',    'format': InsnFmt_I_II, 'size': 'W'},
        {'name': 'chlvl',   'valuemask': '01001011',    'format': InsnFmt_I_II, 'size': 'B'},
        {'name': 'chkar',   'valuemask': '01001101',    'format': InsnFmt_I_II, 'size': ['W','B']},
        {'name': 'chkaw',   'valuemask': '01001110',    'format': InsnFmt_I_II, 'size': ['W','B']},
        {'name': 'chkae',   'valuemask': '01001111',    'format': InsnFmt_I_II, 'size': ['W','B']},
        {'name': 'tasi',    'valuemask': '1110000m',    'format': InsnFmt_III,  'size': 'B'},
        {'name': 'caxi',    'valuemask': '01001100',    'format': InsnFmt_I,    'size': 'W'},
        {'name': 'setf',    'valuemask': '01000111',    'format': InsnFmt_I_II, 'size': 'B'},

        # privileged instructions
        {'name': 'ldpr',    'valuemask': '00010010',    'format': InsnFmt_I_II, 'size': 'W', 'feature': ['IS_LDSTPR']},
        {'name': 'stpr',    'valuemask': '00000010',    'format': InsnFmt_I_II, 'size': 'W', 'feature': ['IS_LDSTPR']},
        {'name': 'clrtlb',  'valuemask': '1111111m',    'format': InsnFmt_III,  'size': 'W'},
        {'name': 'clrtlba', 'valuemask': '00010000',    'format': InsnFmt_V},
        {'name': 'getate',  'valuemask': '00000101',    'format': InsnFmt_I_II, 'size': ['O', 'W']},
        {'name': 'update',  'valuemask': '00010101',    'format': InsnFmt_I_II, 'size': ['O', 'W']},
        {'name': 'getpte',  'valuemask': '00000100',    'format': InsnFmt_I_II, 'size': ['O', 'W']},
        {'name': 'updpte',  'valuemask': '00010100',    'format': InsnFmt_I_II, 'size': ['O', 'W']},
        {'name': 'getra',   'valuemask': '00000011',    'format': InsnFmt_I_II, 'size': ['O', 'W']},
        {'name': 'in',      'valuemask': '00100ss0',    'format': InsnFmt_I_II},
        {'name': 'out',     'valuemask': '00100ss1',    'format': InsnFmt_I_II},
        {'name': 'ldtask',  'valuemask': '00000001',    'format': InsnFmt_I_II, 'size': 'W'},
        {'name': 'sttask',  'valuemask': '1111110m',    'format': InsnFmt_III,  'size': 'W'},
        {'name': 'retis',   'valuemask': '1111101m',    'format': InsnFmt_III,  'size': 'H'},
        {'name': 'updpsw.w','valuemask': '00010011',    'format': InsnFmt_I_II, 'size': 'W'},
        {'name': 'halt',    'valuemask': '00000000',    'format': InsnFmt_V},
    ]

    # Array of instructions
    instruc = [
        {'name': '',  'feature': 0},                                # placeholder for "not an instruction"

        # data transfer instructions
        {'name': 'mov',     'feature': CF_USE1 | CF_CHG2,                       'cmt': "Move source to destination"},
        {'name': 'movs',    'feature': CF_USE1 | CF_CHG2,                       'cmt': "Sign-extend source to destination"},
        {'name': 'xch',     'feature': CF_USE1 | CF_CHG1 | CF_USE2 | CF_CHG2,   'cmt': "Exchange source and destination"},
        {'name': 'movea',   'feature': CF_USE1 | CF_CHG2,                       'cmt': "Move address of source to destination"},
        {'name': 'rvbyt',   'feature': CF_USE1 | CF_CHG2,                       'cmt': "Reverse bytes of source into destination"},
        {'name': 'rvbit',   'feature': CF_USE1 | CF_CHG2,                       'cmt': "Reverse bits of source into destination"},

        # integer arithmetic instructions
        {'name': 'add',     'feature': CF_USE1 | CF_USE2 | CF_CHG2, 'cmt': "Add source to destination"},
        {'name': 'addc',    'feature': CF_USE1 | CF_USE2 | CF_CHG2, 'cmt': "Add source and carry to destination"},
        {'name': 'sub',     'feature': CF_USE1 | CF_USE2 | CF_CHG2, 'cmt': "Subtract source from destination"},
        {'name': 'subc',    'feature': CF_USE1 | CF_USE2 | CF_CHG2, 'cmt': "Subtract source with carry from destination"},
        {'name': 'mul',     'feature': CF_USE1 | CF_USE2 | CF_CHG2, 'cmt': "Signed multiply source by destination"},
        {'name': 'mulu',    'feature': CF_USE1 | CF_USE2 | CF_CHG2, 'cmt': "Unsigned multiply source by destination"},
        {'name': 'mulx',    'feature': CF_USE1 | CF_USE2 | CF_CHG2, 'cmt': "Signed multiply source by extended destination"},
        {'name': 'mulux',   'feature': CF_USE1 | CF_USE2 | CF_CHG2, 'cmt': "Unsigned multiply source by extended destination"},
        {'name': 'div',     'feature': CF_USE1 | CF_USE2 | CF_CHG2, 'cmt': "Signed divide destination by source"},
        {'name': 'divu',    'feature': CF_USE1 | CF_USE2 | CF_CHG2, 'cmt': "Unsigned divide destination by source"},
        {'name': 'divx',    'feature': CF_USE1 | CF_USE2 | CF_CHG2, 'cmt': "Signed divide extended destination by source"},
        {'name': 'divux',   'feature': CF_USE1 | CF_USE2 | CF_CHG2, 'cmt': "Unsigned divide extended destination by source"},
        {'name': 'rem',     'feature': CF_USE1 | CF_USE2 | CF_CHG2, 'cmt': "Get remainder of signed dividing destination by source"},
        {'name': 'remu',    'feature': CF_USE1 | CF_USE2 | CF_CHG2, 'cmt': "Get remainder of unsigned dividing destination by source"},
        {'name': 'inc',     'feature': CF_USE1 | CF_CHG1,           'cmt': "Increment destination"},
        {'name': 'dec',     'feature': CF_USE1 | CF_CHG1,           'cmt': "Decrement destination"},
        {'name': 'neg',     'feature': CF_USE1 | CF_CHG2,           'cmt': "Negate source and store to destination"},
        {'name': 'cmp',     'feature': CF_USE1 | CF_USE2,           'cmt': "Compare source and destination"},
        {'name': 'test',    'feature': CF_USE1,                     'cmt': "Compare source with zero"},

        # logical instructions
        {'name': 'and',     'feature': CF_USE1 | CF_USE2 | CF_CHG2, 'cmt': "Bitwise AND source with destination"},
        {'name': 'or',      'feature': CF_USE1 | CF_USE2 | CF_CHG2, 'cmt': "Bitwise OR source with destination"},
        {'name': 'xor',     'feature': CF_USE1 | CF_USE2 | CF_CHG2, 'cmt': "Exclusive OR source with destination"},
        {'name': 'not',     'feature': CF_USE1 | CF_CHG2,           'cmt': "Bitwise invert source and store to destination"},

        # shift/rotate instructions
        {'name': 'sha',     'feature': CF_USE1 | CF_USE2 | CF_CHG2 | CF_SHFT,   'cmt': "Arithmetically shift destination"},
        {'name': 'shl',     'feature': CF_USE1 | CF_USE2 | CF_CHG2 | CF_SHFT,   'cmt': "Logically shift destination"},
        {'name': 'rot',     'feature': CF_USE1 | CF_USE2 | CF_CHG2 | CF_SHFT,   'cmt': "Rotate destination"},
        {'name': 'rotc',    'feature': CF_USE1 | CF_USE2 | CF_CHG2 | CF_SHFT,   'cmt': "Rotate destination through carry"},

        # floating point instructions
        {'name': 'movf',    'feature': CF_USE1 | CF_CHG2,           'cmt': "Move source to destination"},
        {'name': 'addf',    'feature': CF_USE1 | CF_USE2 | CF_CHG2, 'cmt': "Add source to destination"},
        {'name': 'subf',    'feature': CF_USE1 | CF_USE2 | CF_CHG2, 'cmt': "Subtract source from destination"},
        {'name': 'mulf',    'feature': CF_USE1 | CF_USE2 | CF_CHG2, 'cmt': "Multiply source by destination"},
        {'name': 'divf',    'feature': CF_USE1 | CF_USE2 | CF_CHG2, 'cmt': "Divide destination by source"},
        {'name': 'cmpf',    'feature': CF_USE1 | CF_USE2,           'cmt': "Compare source and destination"},
        {'name': 'negf',    'feature': CF_USE1 | CF_CHG2,           'cmt': "Negate source and store to destination"},
        {'name': 'absf',    'feature': CF_USE1 | CF_CHG2,           'cmt': "Absolute value"},
        {'name': 'sclf',    'feature': CF_USE1 | CF_USE2 | CF_CHG2, 'cmt': "Multiply destination by 2**count"},
        {'name': 'cvt.sl',  'feature': CF_USE1 | CF_CHG2,           'cmt': "Convert between floating point types"},
        {'name': 'cvt.ls',  'feature': CF_USE1 | CF_CHG2,           'cmt': "Convert between floating point types"},
        {'name': 'cvt.ws',  'feature': CF_USE1 | CF_CHG2,           'cmt': "Convert between floating point types"},
        {'name': 'cvt.wl',  'feature': CF_USE1 | CF_CHG2,           'cmt': "Convert between floating point types"},
        {'name': 'cvt.sw',  'feature': CF_USE1 | CF_CHG2,           'cmt': "Convert between floating point types"},
        {'name': 'cvt.lw',  'feature': CF_USE1 | CF_CHG2,           'cmt': "Convert between floating point types"},
        {'name': 'trapfl',  'feature': 0,                           'cmt': "Trap on floating point exception"},

        # decimal arithmetic instructions
        {'name': 'adddc',   'feature': CF_USE1 | CF_USE2 | CF_CHG2 | CF_USE3,   'cmt': "Add decimal source to destination"},
        {'name': 'subdc',   'feature': CF_USE1 | CF_USE2 | CF_CHG2 | CF_USE3,   'cmt': "Subtract decimal source from destination"},
        {'name': 'subrdc',  'feature': CF_USE1 | CF_USE2 | CF_CHG2 | CF_USE3,   'cmt': "Subtract decimal destination from source"},
        {'name': 'cvtd.pz', 'feature': CF_USE1 | CF_USE2 | CF_CHG2 | CF_USE3,   'cmt': "Convert packed to zoned decimal"},
        {'name': 'cvtd.zp', 'feature': CF_USE1 | CF_USE2 | CF_CHG2 | CF_USE3,   'cmt': "Convert zoned to packed decimal"},

        # bit manipulation instructions
        {'name': 'test1',   'feature': CF_USE1 | CF_USE2,           'cmt': "Test bit"},
        {'name': 'set1',    'feature': CF_USE1 | CF_USE2,           'cmt': "Test and set bit"},
        {'name': 'clr1',    'feature': CF_USE1 | CF_USE2,           'cmt': "Test and clear bit"},
        {'name': 'not1',    'feature': CF_USE1 | CF_USE2,           'cmt': "Test and invert bit"},

        # bit field instructions
        # extbfs
        # extbfz
        # extbfl
        # insbfr
        # insbfl
        # cmpbfs
        # cmpbfz
        # cmpbfl

        # bit string instructions
        # movbs
        # notbs
        # andbs
        # andnbs
        # orbs
        # ornbs
        # xorbs
        # xornbs
        # sch0bs
        # sch1bs

        # character manipulation instructions
        # movc
        # movcf
        # movcs
        # cmpc
        # cmpcf
        # cmpcs
        # schc
        # skpc

        # stack manipulation instructions
        {'name': 'push',    'feature': CF_USE1,                     'cmt': "Pop word"},
        {'name': 'pop',     'feature': CF_CHG1,                     'cmt': "Pop word"},
        {'name': 'pushm',   'feature': CF_USE1,                     'cmt': "Push multiple registers"},
        {'name': 'popm',    'feature': CF_USE1,                     'cmt': "Pop multiple registers"},
        {'name': 'prepare', 'feature': CF_USE1,                     'cmt': "Prepare stack frame"},
        {'name': 'dispose', 'feature': 0,                           'cmt': "Dispose stack frame"},

        # control transfer instructions
        {'name': 'bv',      'feature': CF_USE1 | CF_JUMP,           'cmt': "Branch if overflow"},
        {'name': 'bnv',     'feature': CF_USE1 | CF_JUMP,           'cmt': "Branch if no overflow"},
        {'name': 'bc',      'feature': CF_USE1 | CF_JUMP,           'cmt': "Branch if carry (lower)"},
        {'name': 'bnc',     'feature': CF_USE1 | CF_JUMP,           'cmt': "Branch if not carry (not lower)"},
        {'name': 'bz',      'feature': CF_USE1 | CF_JUMP,           'cmt': "Branch if zero (equal)"},
        {'name': 'bnz',     'feature': CF_USE1 | CF_JUMP,           'cmt': "Branch if not zero (not equal)"},
        {'name': 'bnh',     'feature': CF_USE1 | CF_JUMP,           'cmt': "Branch if not higher"},
        {'name': 'bh',      'feature': CF_USE1 | CF_JUMP,           'cmt': "Branch if higher"},
        {'name': 'bn',      'feature': CF_USE1 | CF_JUMP,           'cmt': "Branch if negative"},
        {'name': 'bp',      'feature': CF_USE1 | CF_JUMP,           'cmt': "Branch if positive"},
        {'name': 'br',      'feature': CF_USE1 | CF_JUMP | CF_STOP, 'cmt': "Branch always"},
        {'name': 'blt',     'feature': CF_USE1 | CF_JUMP,           'cmt': "Branch if less"},
        {'name': 'bge',     'feature': CF_USE1 | CF_JUMP,           'cmt': "Branch if greater or equal"},
        {'name': 'ble',     'feature': CF_USE1 | CF_JUMP,           'cmt': "Branch if less or equal"},
        {'name': 'bgt',     'feature': CF_USE1 | CF_JUMP,           'cmt': "Branch if greater"},

        {'name': 'dbv',     'feature': CF_USE1 | CF_USE2 | CF_JUMP, 'cmt': "Decrement and branch if overflow"},
        {'name': 'dbnv',    'feature': CF_USE1 | CF_USE2 | CF_JUMP, 'cmt': "Decrement and branch if no overflow"},
        {'name': 'dbc',     'feature': CF_USE1 | CF_USE2 | CF_JUMP, 'cmt': "Decrement and branch if carry (lower)"},
        {'name': 'dbnc',    'feature': CF_USE1 | CF_USE2 | CF_JUMP, 'cmt': "Decrement and branch if not carry (not lower)"},
        {'name': 'dbz',     'feature': CF_USE1 | CF_USE2 | CF_JUMP, 'cmt': "Decrement and branch if zero (equal)"},
        {'name': 'dbnz',    'feature': CF_USE1 | CF_USE2 | CF_JUMP, 'cmt': "Decrement and branch if not zero (not equal)"},
        {'name': 'dbnh',    'feature': CF_USE1 | CF_USE2 | CF_JUMP, 'cmt': "Decrement and branch if not higher"},
        {'name': 'dbh',     'feature': CF_USE1 | CF_USE2 | CF_JUMP, 'cmt': "Decrement and branch if higher"},
        {'name': 'dbn',     'feature': CF_USE1 | CF_USE2 | CF_JUMP, 'cmt': "Decrement and branch if negative"},
        {'name': 'dbp',     'feature': CF_USE1 | CF_USE2 | CF_JUMP, 'cmt': "Decrement and branch if positive"},
        {'name': 'dbr',     'feature': CF_USE1 | CF_USE2 | CF_JUMP, 'cmt': "Decrement and branch"},
        {'name': 'dblt',    'feature': CF_USE1 | CF_USE2 | CF_JUMP, 'cmt': "Decrement and branch if less"},
        {'name': 'dbge',    'feature': CF_USE1 | CF_USE2 | CF_JUMP, 'cmt': "Decrement and branch if greater or equal"},
        {'name': 'dble',    'feature': CF_USE1 | CF_USE2 | CF_JUMP, 'cmt': "Decrement and branch if less or equal"},
        {'name': 'dbgt',    'feature': CF_USE1 | CF_USE2 | CF_JUMP, 'cmt': "Decrement and branch if greater"},

        {'name': 'tb',      'feature': CF_USE1 | CF_USE2 | CF_JUMP, 'cmt': "Test and branch"},
        {'name': 'jmp',     'feature': CF_USE1 | CF_CALL | CF_STOP, 'cmt': "Jump"},
        {'name': 'bsr',     'feature': CF_USE1 | CF_CALL,           'cmt': "Branch to subroutine"},
        {'name': 'jsr',     'feature': CF_USE1 | CF_CALL,           'cmt': "Jump to subroutine"},
        {'name': 'rsr',     'feature': CF_STOP,                     'cmt': "Return from subroutine"},
        {'name': 'call',    'feature': CF_USE1 | CF_USE2 | CF_CALL, 'cmt': "Call procedure"},
        {'name': 'ret',     'feature': CF_USE1 | CF_STOP,           'cmt': "Return from procedure"},
        {'name': 'brk',     'feature': 0,                           'cmt': "Breakpoint"},
        {'name': 'brkv',    'feature': 0,                           'cmt': "Break on overflow"},
        {'name': 'trap',    'feature': CF_USE1,                     'cmt': "Trap on condition"},
        {'name': 'retiu',   'feature': CF_USE1 | CF_STOP,           'cmt': "Return from interrupt (user)"},

        # miscellaneous instructions
        {'name': 'nop',     'feature': 0,                           'cmt': "No operation"},
        {'name': 'getpsw',  'feature': CF_CHG1,                     'cmt': "Get program status word"},
        {'name': 'updpsw.h','feature': CF_USE1 | CF_USE2,           'cmt': "Update program status word condition codes"},
        {'name': 'chlvl',   'feature': CF_USE1 | CF_USE2,           'cmt': "Change execution level"},
        {'name': 'chkar',   'feature': CF_USE1 | CF_USE2,           'cmt': "Check read access permission"},
        {'name': 'chkaw',   'feature': CF_USE1 | CF_USE2,           'cmt': "Check write access permission"},
        {'name': 'chkae',   'feature': CF_USE1 | CF_USE2,           'cmt': "Check execute access permission"},
        {'name': 'tasi',    'feature': CF_USE1 | CF_CHG1,           'cmt': "Test and set interlocked"},
        {'name': 'caxi',    'feature': CF_USE1 | CF_CHG1 | CF_USE2 | CF_CHG2,   'cmt': "Compare and exchange interlocked"},
        {'name': 'setf',    'feature': CF_USE1 | CF_CHG2,           'cmt': "Set if condition is true"},

        # privileged instructions
        {'name': 'ldpr',    'feature': CF_USE1 | CF_USE2,           'cmt': "Load privileged register"},
        {'name': 'stpr',    'feature': CF_USE1 | CF_USE2,           'cmt': "Store privileged register"},
        {'name': 'clrtlb',  'feature': CF_USE1,                     'cmt': "Clear TLB entry"},
        {'name': 'clrtlba', 'feature': CF_USE1,                     'cmt': "Clear all TLB entries"},
        {'name': 'getate',  'feature': CF_USE1 | CF_CHG2,           'cmt': "Get area table entry"},
        {'name': 'update',  'feature': CF_USE1 | CF_USE2,           'cmt': "Update area table entry"},
        {'name': 'getpte',  'feature': CF_USE1 | CF_CHG2,           'cmt': "Get page table entry"},
        {'name': 'updpte',  'feature': CF_USE1 | CF_USE2,           'cmt': "Update page table entry"},
        {'name': 'getra',   'feature': CF_USE1 | CF_CHG2,           'cmt': "Get physical address"},
        {'name': 'in',      'feature': CF_USE1 | CF_CHG2,           'cmt': "Read from input port"},
        {'name': 'out',     'feature': CF_USE1 | CF_USE2,           'cmt': "Write to output port"},
        {'name': 'ldtask',  'feature': CF_USE1 | CF_USE2,           'cmt': "Load task context"},
        {'name': 'sttask',  'feature': CF_USE1,                     'cmt': "Store task context"},
        {'name': 'retis',   'feature': CF_USE1 | CF_STOP,           'cmt': "Return from interrupt (system)"},
        {'name': 'updpsw.w','feature': CF_USE1 | CF_USE2,           'cmt': "Update program status word"},
        {'name': 'halt',    'feature': CF_STOP,                     'cmt': "Halt"},
    ]

    # icode of the first instruction
    instruc_start = 0

    # icode of the last instruction + 1
    instruc_end = len(instruc) + 1

    # Size of long double (tbyte) for this processor (meaningful only if ash.a_tbyte != NULL) (optional)
    # tbyte_size = 0

    #
    # Number of digits in floating numbers after the decimal point.
    # If an element of this array equals 0, then the corresponding
    # floating point data is not used for the processor.
    # This array is used to align numbers in the output.
    #      real_width[0] - number of digits for short floats (only PDP-11 has them)
    #      real_width[1] - number of digits for "float"
    #      real_width[2] - number of digits for "double"
    #      real_width[3] - number of digits for "long double"
    # Example: IBM PC module has { 0,7,15,19 }
    #
    # (optional)
    real_width = (0, 7,15, 0)


    # only one assembler is supported
    assembler = {
        # flag
        'flag' : ASH_HEXF0 | ASD_DECF0 | ASO_OCTF5 | ASB_BINF0 | AS_N2CHR,

        # user defined flags (local only for IDP) (optional)
        'uflag' : 0,

        # Assembler name (displayed in menus)
        'name': "NEC V60 assembler",

        # array of automatically generated header lines they appear at the start of disassembled text (optional)
        'header': ['# V60'],

        # org directive
        'origin': ".org",

        # end directive
        'end': ".end",

        # comment string (see also cmnt2)
        'cmnt': ";",

        # ASCII string delimiter
        'ascsep': "\"",

        # ASCII char constant delimiter
        'accsep': "'",

        # ASCII special chars (they can't appear in character and ascii constants)
        'esccodes': "\"'",

        #
        #      Data representation (db,dw,...):
        #
        # ASCII string directive
        'a_ascii': ".ascii",

        # byte directive
        'a_byte': ".byte",

        # word directive
        'a_word': ".half",

        # remove if not allowed
        'a_dword': ".word",

        # remove if not allowed
        'a_qword': ".dword",

        # float;  4bytes; remove if not allowed
        'a_float': ".float",

        # uninitialized data directive (should include '%s' for the size of data)
        'a_bss': ".space %s",

        # 'equ' Used if AS_UNEQU is set (optional)
        'a_equ': ".equ",

        # 'seg ' prefix (example: push seg seg001)
        'a_seg': "seg",

        # current IP (instruction pointer) symbol in assembler
        'a_curip': "$",

        # "public" name keyword. NULL-gen default, ""-do not generate
        'a_public': ".global",

        # "weak"   name keyword. NULL-gen default, ""-do not generate
        'a_weak': "",

        # "extrn"  name keyword
        'a_extrn': ".extern",

        # "comm" (communal variable)
        'a_comdef': "",

        # "align" keyword
        'a_align': ".align",

        # Left and right braces used in complex expressions
        'lbrace': "(",
        'rbrace': ")",

        # %  mod     assembler time operation
        'a_mod': "%",

        # &  bit and assembler time operation
        'a_band': "&",

        # |  bit or  assembler time operation
        'a_bor': "|",

        # ^  bit xor assembler time operation
        'a_xor': "^",

        # ~  bit not assembler time operation
        'a_bnot': "~",

        # << shift left assembler time operation
        'a_shl': "<<",

        # >> shift right assembler time operation
        'a_shr': ">>",

        # size of type (format string) (optional)
        'a_sizeof_fmt': "size %s",

        'flag2': 0,

        # the include directive (format string) (optional)
        'a_include_fmt': '.include "%s"',
    } # Assembler


    # ----------------------------------------------------------------------
    # The following callbacks are optional
    #

    #def notify_newprc(self, nproc, keep_cfg):
    #    """
    #    Before changing proccesor type
    #    nproc - processor number in the array of processor names
    #    return 1-ok,0-prohibit
    #    """
    #    return 1

    #def notify_assemble(self, ea, cs, ip, use32, line):
    #    """
    #    Assemble an instruction
    #     (make sure that PR_ASSEMBLE flag is set in the processor flags)
    #     (display a warning if an error occurs)
    #     args:
    #       ea -  linear address of instruction
    #       cs -  cs of instruction
    #       ip -  ip of instruction
    #       use32 - is 32bit segment?
    #       line - line to assemble
    #    returns the opcode string
    #    """
    #    pass

    #def notify_get_frame_retsize(self, func_ea):
    #    """
    #    Get size of function return address in bytes
    #    If this function is absent, the kernel will assume
    #         4 bytes for 32-bit function
    #         2 bytes otherwise
    #    """
    #    return 2

    def notify_get_autocmt(self, insn):
        """
        Get instruction comment. 'insn' describes the instruction in question
        @return: None or the comment string
        """
        if 'cmt' in self.instruc[insn.itype]:
          return self.instruc[insn.itype]['cmt']

    # ----------------------------------------------------------------------
    def handle_operand(self, insn, op, is_read):
        flags     = get_flags(insn.ea)
        is_offs   = is_off(flags, op.n)
        dref_flag = dr_R if is_read else dr_W
        if insn.itype == self.itype_movea:
            dref_flag = dr_O

        if op.type == o_imm:
            if is_offs:
                insn.add_off_drefs(op, dr_O, 0)
        elif op.type == o_displ:
            # delta[reg]
            if is_offs:
                insn.add_off_drefs(op, dref_flag, OOF_ADDR)
        elif op.type == o_mem:
            # create data xrefs
            insn.create_op_data(op.addr, op)
            insn.add_dref(op.addr, op.offb, dref_flag)
        elif op.type == o_near:
            # create code xrefs
            if insn.get_canon_feature() & CF_CALL:
                fl = fl_CN
            else:
                fl = fl_JN
            insn.add_cref(op.addr, op.offb, fl)

    # ----------------------------------------------------------------------
    def add_stkpnt(self, pfn, insn, v):
        if pfn:
            end = insn.ea + insn.size
            if not is_fixed_spd(end):
                ida_frame.add_auto_stkpnt(pfn, end, v)

    # ----------------------------------------------------------------------
    # The following callbacks are mandatory
    #
    def notify_emu(self, insn):
        """
        Emulate instruction, create cross-references, plan to analyze
        subsequent instructions, modify flags etc. Upon entrance to this function
        all information about the instruction is in 'insn' structure.
        If zero is returned, the kernel will delete the instruction.
        """
        aux = self.get_auxpref(insn)
        Feature = insn.get_canon_feature()

        if Feature & CF_USE1:
            self.handle_operand(insn, insn.Op1, 1)
        if Feature & CF_CHG1:
            self.handle_operand(insn, insn.Op1, 0)
        if Feature & CF_USE2:
            self.handle_operand(insn, insn.Op2, 1)
        if Feature & CF_CHG2:
            self.handle_operand(insn, insn.Op2, 0)
        if Feature & CF_USE3:
            self.handle_operand(insn, insn.Op3, 1)
        if Feature & CF_CHG3:
            self.handle_operand(insn, insn.Op3, 0)
        if Feature & CF_USE4:
            self.handle_operand(insn, insn.Op4, 1)
        if Feature & CF_CHG4:
            self.handle_operand(insn, insn.Op4, 0)
        if Feature & CF_JUMP:
            remember_problem(PR_JUMP, insn.ea)
        if Feature & CF_CALL:
            remember_problem(PR_JUMP, insn.ea)

        # add flow
        if (Feature & CF_STOP) == 0:
            add_cref(insn.ea, insn.ea + insn.size, fl_F)

        return 1

    # ----------------------------------------------------------------------
    def emit_direct_addr(self, ctx, op):
        r = ctx.out_name_expr(op, op.addr, BADADDR)
        if not r:
            ctx.out_tagon(COLOR_ERROR)
            ctx.out_btoa(op.addr, 16)
            ctx.out_tagoff(COLOR_ERROR)
            remember_problem(PR_NONAME, ctx.insn.ea)

    def notify_out_operand(self, ctx, op):
        """
        Generate text representation of an instructon operand.
        This function shouldn't change the database, flags or anything else.
        All these actions should be performed only by the emu() function.
        This function uses out_...() functions from ua.hpp to generate the operand text
        Returns: 1-ok, 0-operand is hidden.
        """
        optype = op.type
        am = op.specval & ~AddrModeFlagMask
        signed = 0 if op.specflag2 == 0 else OOF_SIGNED

        if optype == o_reg:
            ctx.out_register(self.reg_names[op.reg])
        elif optype == o_imm:
            ctx.out_symbol('#')
            if ImmediatesAreSignExtended:
                ctx.out_value(op, OOF_SIGNED)
            else:
                ctx.out_value(op, signed)
        elif optype == o_mem or optype == o_near or optype == o_displ:
            disp_flags = OOF_SIGNED | OOFW_32
            if am == AddrMode_Displacement:
                if op.type == o_displ: # Not resolved displacement against PC
                    ctx.out_value(op, OOF_ADDR | disp_flags)
                else:
                    self.emit_direct_addr(ctx, op)
                ctx.out_symbol('[')
                ctx.out_register(self.reg_names[op.reg])
                ctx.out_symbol(']')
            elif am == AddrMode_DisplacementIndirect:
                ctx.out_symbol('[')
                if op.type == o_displ: # Not resolved displacement against PC
                    ctx.out_value(op, OOF_ADDR | disp_flags)
                else:
                    self.emit_direct_addr(ctx, op)
                ctx.out_symbol('[')
                ctx.out_register(self.reg_names[op.reg])
                ctx.out_symbol(']')
                ctx.out_symbol(']')
            elif am == AddrMode_DoubleDisplacement:
                ctx.out_value(op, disp_flags)
                ctx.out_symbol('[')
                if op.type == o_displ: # Not resolved displacement against PC
                    ctx.out_value(op, OOF_ADDR | disp_flags)
                else:
                    self.emit_direct_addr(ctx, op)
                ctx.out_symbol('[')
                ctx.out_register(self.reg_names[op.reg])
                ctx.out_symbol(']')
                ctx.out_symbol(']')
            elif am == AddrMode_DirectAddress:
                self.emit_direct_addr(ctx, op)
            elif am == AddrMode_DirectAddressDeferred:
                ctx.out_symbol('[')
                self.emit_direct_addr(ctx, op)
                ctx.out_symbol(']')
            else:
                raise Exception("Unhandled address mode %d for memory optype" % am)
        elif optype == o_phrase:
            if am == AddrMode_RegisterIndirect:
                ctx.out_symbol('[')
                ctx.out_register(self.reg_names[op.reg])
                ctx.out_symbol(']')
            elif am == AddrMode_Autoincrement:
                ctx.out_symbol('[')
                ctx.out_register(self.reg_names[op.reg])
                ctx.out_symbol('+')
                ctx.out_symbol(']')
            elif am == AddrMode_Autodecrement:
                ctx.out_symbol('[')
                ctx.out_symbol('-')
                ctx.out_register(self.reg_names[op.reg])
                ctx.out_symbol(']')
            else:
                raise Exception("Unhandled address mode %d for memory optype" % am)
        else:
            ctx.out_tagon(COLOR_ERROR)
            ctx.out_symbol('T')
            ctx.out_symbol('O')
            ctx.out_symbol('D')
            ctx.out_symbol('O')
            ctx.out_tagoff(COLOR_ERROR)
            raise Exception("Unhandled printing address mode %d" % am)
            return True

        # Indexed suffix
        if (op.specval & AddrModeFlag_Indexed) != 0:
            ctx.out_symbol('(')
            ctx.out_register(self.reg_names[op.specflag1])
            ctx.out_symbol(')')

        return True

    # ----------------------------------------------------------------------
    def out_mnem(self, ctx):
        # add postfix if necessary
        sz = ctx.insn.auxpref & InsnAux_Postfix_Mask
        postfix = ("",
                   ".b",
                   ".h",
                   ".w",
                   ".d",
                   ".bh",
                   ".bw",
                   ".hb",
                   ".hw",
                   ".wb",
                   ".wh",
                  )[sz]

        # change to movz or movt
        if ctx.insn.itype == self.itype_mov and sz >= InsnAux_Postfix_BH:
            if sz in (InsnAux_Postfix_BH, InsnAux_Postfix_BW, InsnAux_Postfix_HW):
                postfix = 'z' + postfix
            elif sz in (InsnAux_Postfix_HB, InsnAux_Postfix_WB, InsnAux_Postfix_WH):
                postfix = 't' + postfix

        # first argument (8) is the width of the mnemonic field
        ctx.out_mnem(8, postfix)

    # ----------------------------------------------------------------------
    def notify_out_insn(self, ctx):
        """
        Generate text representation of an instruction in 'ctx.insn' structure.
        This function shouldn't change the database, flags or anything else.
        All these actions should be performed only by emu() function.
        Returns: nothing
        """
        ctx.out_mnemonic()

        # output first operand
        # kernel will call out_operand()
        if ctx.insn.Op1.type != o_void:
            ctx.out_one_operand(0)

        # output the rest of operands separated by commas
        for i in xrange(1, 3):
            if ctx.insn[i].type == o_void:
                break
            ctx.out_symbol(',')
            ctx.out_char(' ')
            ctx.out_one_operand(i)

        ctx.set_gen_cmt() # generate comment at the next call to MakeLine()
        ctx.flush_outbuf()

    # ----------------------------------------------------------------------
    def decode_register(self, insn, op, reg, size, force_signed = False):
        op.specval = AddrMode_Register
        op.type = o_reg
        op.reg = reg
        op.dtype = size
        op.specflag2 = 1 if force_signed else 0
        return True

    def displacement_resolve(self, insn, op):
        if (op.specval & ~AddrModeFlagMask) in (AddrMode_Displacement,
                                                AddrMode_DisplacementIndirect,
                                                AddrMode_DoubleDisplacement):
            if op.reg == self.ireg_PC:
                op.type = o_mem
                op.addr = ZEROEXT(insn.ea + op.addr, 32)
                return True
        return False

    def decode_jump_displacement(self, insn, op, size):
        op.specval = AddrMode_Displacement
        op.type = o_displ
        op.reg = self.ireg_PC
        op.dtype = size
        op.addr = self.disp_fetch(insn, dtype_size_bits[size])
        if self.displacement_resolve(insn, op):
            op.type = o_near
        return True


    def fetch(self, insn, size):
        v = 0
        for i in range(size / 8):
            v |= insn.get_next_byte() << (i * 8)
        return v

    def disp_fetch(self, insn, size):
        return DISPEXT(self.fetch(insn, size), size)
    def imm_fetch(self, insn, size):
        return IMMEXT(self.fetch(insn, size), size)

    def decode_ext(self, insn, op):
        ext = insn.get_next_byte()
        if BIT(ext, 7) == 0: # Direct mode
            op.specval = AddrMode_Immediate
            op.type = o_imm
            op.addr = op.value = BITS(ext, 6, 0)
        else: # Indirect mode
            op.specval = AddrMode_Register
            op.type = o_reg
            op.reg = BITS(ext, 4, 0)
        return True

    def decode_imm8(self, insn, op):
        op.specval = AddrMode_Immediate
        op.type = o_imm
        op.addr = op.value = insn.get_next_byte()
        op.dtype = dt_byte
        return True

    def decode_mod(self, insn, op, m, size, insndat, force_signed = False):
        op.dtype = size
        op.specflag2 = 1 if force_signed else 0
        mod = ZEROEXT(insn.get_next_byte(), 8) | (m << 8)
        op.reg = BITS(mod, 4, 0)
        am = 0

        if BITS(mod, 8, 5) == 0b1110: # Indexed prefix
            am = AddrModeFlag_Indexed
            op.specflag1 = op.reg
            mod = insn.get_next_byte()
            op.reg = BITS(mod, 4, 0)

        if BITS(mod, 8, 5) == 0b1011: # Rn
            am |= AddrMode_Register
            op.type = o_reg
        elif BITS(mod, 8, 5) == 0b0011: # [Rn]
            am |= AddrMode_RegisterIndirect
            op.type = o_phrase
        elif BITS(mod, 8, 5) == 0b1100: # [Rn+]
            am |= AddrMode_Autoincrement
            op.type = o_phrase
        elif BITS(mod, 8, 5) == 0b1101: # [-Rn]
            am |= AddrMode_Autoincrement
            op.type = o_phrase
        elif BITS(mod, 8, 4) == 0b01110: # immed.4
            am |= AddrMode_Immediate
            op.type = o_imm
            op.addr = op.value = BITS(mod, 3, 0)
        elif BITS(mod, 8, 0) == 0b011110100: # immed.size
            am |= AddrMode_Immediate
            op.type = o_imm
            op.addr = op.value = self.imm_fetch(insn, dtype_size_bits[size])
        elif BITS(mod, 8, 5) == 0b0000: # disp.8[Rn]
            am |= AddrMode_Displacement
            op.type = o_displ
            op.addr = self.disp_fetch(insn, 8)
        elif BITS(mod, 8, 5) == 0b0001: # disp.16[Rn]
            am |= AddrMode_Displacement
            op.type = o_displ
            op.addr = self.disp_fetch(insn, 16)
        elif BITS(mod, 8, 5) == 0b0010: # disp.32[Rn]
            am |= AddrMode_Displacement
            op.type = o_displ
            op.addr = self.disp_fetch(insn, 32)
        elif BITS(mod, 8, 0) == 0b011110000: # disp.8[PC]
            am |= AddrMode_Displacement
            op.type = o_displ
            op.reg = self.ireg_PC
            op.addr = self.disp_fetch(insn, 8)
        elif BITS(mod, 8, 0) == 0b011110001: # disp.16[PC]
            am |= AddrMode_Displacement
            op.type = o_displ
            op.reg = self.ireg_PC
            op.addr = self.disp_fetch(insn, 16)
        elif BITS(mod, 8, 0) == 0b011110010: # disp.32[PC]
            am |= AddrMode_Displacement
            op.type = o_displ
            op.reg = self.ireg_PC
            op.addr = self.disp_fetch(insn, 32)
        elif BITS(mod, 8, 5) == 0b0100: # [disp.8[Rn]]
            am |= AddrMode_DisplacementIndirect
            op.type = o_displ
            op.addr = self.disp_fetch(insn, 8)
        elif BITS(mod, 8, 5) == 0b0101: # [disp.16[Rn]]
            am |= AddrMode_DisplacementIndirect
            op.type = o_displ
            op.addr = self.disp_fetch(insn, 16)
        elif BITS(mod, 8, 5) == 0b0110: # [disp.32[Rn]]
            am |= AddrMode_DisplacementIndirect
            op.type = o_displ
            op.addr = self.disp_fetch(insn, 32)
        elif BITS(mod, 8, 0) == 0b011111000: # [disp.8[PC]]
            am |= AddrMode_DisplacementIndirect
            op.type = o_displ
            op.reg = self.ireg_PC
            op.addr = self.disp_fetch(insn, 8)
        elif BITS(mod, 8, 0) == 0b011111001: # [disp.16[PC]]
            am |= AddrMode_DisplacementIndirect
            op.type = o_displ
            op.reg = self.ireg_PC
            op.addr = self.disp_fetch(insn, 16)
        elif BITS(mod, 8, 0) == 0b011111010: # [disp.32[PC]]
            am |= AddrMode_DisplacementIndirect
            op.type = o_displ
            op.reg = self.ireg_PC
            op.addr = self.disp_fetch(insn, 32)
        elif BITS(mod, 8, 5) == 0b1000: # disp1.8[disp2.8[Rn]]
            am |= AddrMode_DoubleDisplacement
            op.type = o_displ
            op.addr = self.disp_fetch(insn, 8)
            op.value = self.disp_fetch(insn, 8)
        elif BITS(mod, 8, 5) == 0b1001: # disp1.16[disp2.16[Rn]]
            am |= AddrMode_DoubleDisplacement
            op.type = o_displ
            op.addr = self.disp_fetch(insn, 16)
            op.value = self.disp_fetch(insn, 16)
        elif BITS(mod, 8, 5) == 0b1010: # disp1.32[disp2.32[Rn]]
            am |= AddrMode_DoubleDisplacement
            op.type = o_displ
            op.addr = self.disp_fetch(insn, 32)
            op.value = self.disp_fetch(insn, 32)
        elif BITS(mod, 8, 0) == 0b011111100: # disp1.8[disp2.8[PC]]
            am |= AddrMode_DoubleDisplacement
            op.type = o_displ
            op.reg = self.ireg_PC
            op.addr = self.disp_fetch(insn, 8)
            op.value = self.disp_fetch(insn, 8)
        elif BITS(mod, 8, 0) == 0b011111101: # disp1.16[disp2.16[PC]]
            am |= AddrMode_DoubleDisplacement
            op.type = o_displ
            op.reg = self.ireg_PC
            op.addr = self.disp_fetch(insn, 16)
            op.value = self.disp_fetch(insn, 16)
        elif BITS(mod, 8, 0) == 0b011111110: # disp1.32[disp2.32[PC]]
            am |= AddrMode_DoubleDisplacement
            op.type = o_displ
            op.reg = self.ireg_PC
            op.addr = self.disp_fetch(insn, 32)
            op.value = self.disp_fetch(insn, 32)
        elif BITS(mod, 8, 0) == 0b011110011: # /addr
            am |= AddrMode_DirectAddress
            op.type = o_mem
            op.addr = ZEROEXT(self.fetch(insn, 32), 32)
        elif BITS(mod, 8, 0) == 0b011111011: # [/addr]
            am |= AddrMode_DirectAddressDeferred
            op.type = o_mem
            op.addr = ZEROEXT(self.fetch(insn, 32), 32)
        else:
            if VerboseFailure:
                raise Exception("Unknown mod %X! (@%X, insn=%s)" % (mod, insn.ea, insndat['name']))
            return False
        op.specval = am
        self.displacement_resolve(insn, op)
        # Only a raw displacement actually points to code
        if ((op.type == o_mem) and
            (am not in (AddrMode_DisplacementIndirect,
                        AddrMode_DisplacementIndirect | AddrModeFlag_Indexed,
                        AddrMode_DoubleDisplacement,
                        AddrMode_DoubleDisplacement | AddrModeFlag_Indexed,
                        AddrMode_Displacement | AddrModeFlag_Indexed,
                        )) and
            ('feature' in insndat and 'CODE_DST' in insndat['feature'])):
            op.type = o_near
        return True

    def decode_param_size(self, value):
        return datalen_dtype_mapping[value]

    def generate_postfix(self, insn, insndat, param):
        if 'size' in insndat or ('feature' in insndat and 'NO_SUFFIX' in insndat['feature']):
            return InsnAux_Postfix_None

        first_param = None
        second_param = None
        if 's' in param:
            first_param = param['s']
        if 'S' in param:
            second_param = param['S']

        if first_param == DataLen_Offset:
            first_param = None
        if second_param == DataLen_Offset:
            second_param = None

        if first_param == second_param:
            second_param = None
        if first_param == None:
            first_param = second_param
            second_param = None

        if first_param is not None:
            if second_param is not None:
                if first_param >= DataLen_Dword or second_param >= DataLen_Dword:
                    print("%s at %X" % (insndat['name'], insn.ea))
                    raise Exception("parameter types are invalid? %d and %d" % (first_param, second_param))
                return ((InsnAux_Postfix_B,  InsnAux_Postfix_HB, InsnAux_Postfix_WB,),
                        (InsnAux_Postfix_BH, InsnAux_Postfix_H,  InsnAux_Postfix_WH,),
                        (InsnAux_Postfix_BW, InsnAux_Postfix_HW, InsnAux_Postfix_W, ),
                       )[first_param][second_param]
            else:
                return (InsnAux_Postfix_B,
                        InsnAux_Postfix_H,
                        InsnAux_Postfix_W,
                        InsnAux_Postfix_D,
                       )[first_param]
        else:
            return InsnAux_Postfix_None
    # ----------------------------------------------------------------------
    def decode_fmt_I_raw(self, insn, op, subop, insndat, param):
        #  Format I
        #
        #         15 14 13 12  8 7      0
        #  +-----+--+--+--+-----+--------+
        #  | mod | 0| m| d| reg |   op   |
        #  +-----+--------+-----+--------+
        rm = subop
        reg  = BITS(rm, 12-8, 8-8)
        d    = BIT( rm, 13-8)
        m    = BIT( rm, 14-8)
        zero = BIT( rm, 15-8)
        #if zero != 0:
        #    return False

        if 's' in param:
            size_a = self.decode_param_size(param['s'])
            size_b = size_a
            if 'S' in param:
                size_b = self.decode_param_size(param['S'])
        else:
            # Need a size specifier from somewhere
            raise Exception("Unspecified size")
            return False

        force_signed = False
        if 'feature' in insndat and 'IS_SHIFT' in insndat['feature']:
            size_a = dt_byte
            force_signed = True

        if d == 0:
            if ((not self.decode_register(insn, insn.Op1, reg, size_a, force_signed)) or
                (not self.decode_mod(insn, insn.Op2, m, size_b, insndat))):
                return False
        else:
            if ((not self.decode_mod(insn, insn.Op1, m, size_a, insndat, force_signed)) or
                (not self.decode_register(insn, insn.Op2, reg, size_b))):
                return False

        if 'feature' in insndat and 'IS_LDSTPR' in insndat['feature']:
            if insn.Op2.type == o_imm:
                # TODO: convert to enum
                pass

        insn.auxpref |= self.generate_postfix(insn, insndat, param)
        insn.itype = getattr(self, 'itype_' + insndat['name'])
        return True

    def decode_fmt_I(self, insn, op, subop, insndat, param):
        subop = insn.get_next_byte()
        return self.decode_fmt_I_raw(insn, op, subop, insndat, param)

    # ----------------------------------------------------------------------
    def decode_fmt_II_raw(self, insn, op, rm, insndat, param):
        #  Format II
        #
        #                15 14 13 12  8 7      0
        #  +------+-----+--+--+--+-----+--------+
        #  | mod' | mod | 1| m|m'|subop|   op   |
        #  +------+-----+--------+-----+--------+

        subop= BITS(rm, 12-8, 8-8)
        m2   = BIT( rm, 13-8)
        m    = BIT( rm, 14-8)
        one  = BIT( rm, 15-8)
        #if one != 1:
        #    return False

        if 's' in param:
            size_a = self.decode_param_size(param['s'])
            size_b = size_a
            if 'S' in param:
                size_b = self.decode_param_size(param['S'])
        else:
            # Need a size specifier from somewhere
            raise Exception("Unspecified size")
            return False

        force_signed = False
        if 'feature' in insndat and 'IS_SHIFT' in insndat['feature']:
            size_a = dt_byte
            force_signed = True

        if ((not self.decode_mod(insn, insn.Op1, m, size_a, insndat, force_signed)) or
            (not self.decode_mod(insn, insn.Op2, m2, size_b, insndat))):
            return False

        if 'feature' in insndat and 'IS_LDSTPR' in insndat['feature']:
            if insn.Op2.type == o_imm:
                # TODO: convert to enum
                pass

        insn.auxpref |= self.generate_postfix(insn, insndat, param)
        insn.itype = getattr(self, 'itype_' + insndat['name'])
        return True

    def decode_fmt_II(self, insn, op, subop, insndat, param):
        subop = insn.get_next_byte()
        return self.decode_fmt_II_raw(insn, op, subop, insndat, param)

    # ----------------------------------------------------------------------
    def decode_fmt_I_II(self, insn, op, subop, insndat, param):
        subop = insn.get_next_byte()
        if BIT(subop, 7) == 0:
            return self.decode_fmt_I_raw(insn, op, subop, insndat, param)
        else:
            return self.decode_fmt_II_raw(insn, op, subop, insndat, param)

    # ----------------------------------------------------------------------
    def decode_fmt_III(self, insn, op, subop, insndat, param):
        #  Format III
        #
        #       8 7    1 0
        #  +-----+------+-+
        #  | mod |  op  |m|
        #  +-----+------+-+

        if 's' in param:
            size_a = self.decode_param_size(param['s'])
        else:
            # Need a size specifier from somewhere
            raise Exception("Unspecified size")
            return False
        if 'm' not in param:
            raise Exception("Missing 'm' parameter")

        if not self.decode_mod(insn, insn.Op1, param['m'], size_a, insndat):
            return False

        insn.auxpref |= self.generate_postfix(insn, insndat, param)
        insn.itype = getattr(self, 'itype_' + insndat['name'])
        return True

    # ----------------------------------------------------------------------
    def decode_fmt_IV(self, insn, op, subop, insndat, param):
        #  Format IV
        #
        #   15     8 7     0
        #  +-------+-------+
        #  | disp8 |   op  |
        #  +-------+-------+
        #
        #   23              8 7     0
        #  +-----------------+-------+
        #  |      disp16     |   op  |
        #  +-----------------+-------+

        if 's' in param:
            size = self.decode_param_size(param['s'])
        else:
            # Need a size specifier from somewhere
            raise Exception("Unspecified size")
            return False

        if not self.decode_jump_displacement(insn, insn.Op1, size):
            return False

        insn.auxpref |= self.generate_postfix(insn, insndat, param)
        insn.itype = getattr(self, 'itype_' + insndat['name'])
        return True

    # ----------------------------------------------------------------------
    def decode_fmt_V(self, insn, op, subop, insndat, param):
        #  Format V
        #
        #   7      0
        #  +--------+
        #  |   op   |
        #  +--------+
        insn.itype = getattr(self, 'itype_' + insndat['name'])
        return True

    # ----------------------------------------------------------------------
    def decode_fmt_VI(self, insn, op, subop, insndat, param):
        #  Format VI
        #
        #   31        16 15 13 12  8 7     0
        #  +------------+-----+-----+--------+
        #  |   disp16   |subop| reg |   op   |
        #  +------------+-----+-----+--------+
        reg   = BITS(subop, 12-8, 8-8)

        if ((not self.decode_register(insn, insn.Op1, reg, dt_dword)) or
            (not self.decode_jump_displacement(insn, insn.Op2, dt_word))):
            return False

        insn.itype = getattr(self, 'itype_' + insndat['name'])
        return True

    # ----------------------------------------------------------------------
    def decode_fmt_VII(self, insn, op, rm, insndat, param, subfmt):
        #  Format VII
        #  ext:
        #   7 6      0
        #  +-+--------+
        #  |r| length |
        #  +-+--------+
        #
        #  Format VIIa
        #
        #  +7   +0       +7  +0       15 14 13 12  8 7      0
        #  +------+------+-----+-----+--+--+--+-----+--------+
        #  | ext' | mod' | ext | mod | 1| m|m'|subop|   op   |
        #  +------+------+-----+-----+--------+-----+--------+
        #
        #  Format VIIb
        #
        #         +7  +0       15 14 13 12  8 7      0
        #  +------+-----+-----+--+--+--+-----+--------+
        #  | mod' | ext | mod | 1| m|m'|subop|   op   |
        #  +------+-----+-----+--------+-----+--------+
        #
        #  Format VIIc
        #
        #  +7   +0              15 14 13 12  8 7      0
        #  +------+------+-----+--+--+--+-----+--------+
        #  | ext' | mod' | mod | 1| m|m'|subop|   op   |
        #  +------+------+-----+--------+-----+--------+
        #
        #  Format VIId (VIIc for BCD)
        #
        #  +7   +0              15 14 13 12  8 7      0
        #  +------+------+-----+--+--+--+-----+--------+
        #  | imm8 | mod' | mod | 1| m|m'|subop|   op   |
        #  +------+------+-----+--------+-----+--------+

        subop= BITS(rm, 12-8, 8-8)
        m2   = BIT( rm, 13-8)
        m    = BIT( rm, 14-8)
        one  = BIT( rm, 15-8)

        if 's' in param:
            size_a = self.decode_param_size(param['s'])
            size_b = size_a
            if 'S' in param:
                size_b = self.decode_param_size(param['S'])
        else:
            # Need a size specifier from somewhere
            raise Exception("Unspecified size")
            return False

        if not self.decode_mod(insn, insn.Op1, m, size_a, insndat):
            return False
        if subfmt == 0 or subfmt == 1: # VIIa or VIIb
            if ((not self.decode_ext(insn, insn.Op2)) or
                (not self.decode_mod(insn, insn.Op3, m2, size_b, insndat))):
                return False
            if subfmt == 1: # VIIb
                if not self.decode_ext(insn, insn.Op4):
                    return False
        elif subfmt == 2: # VIIc
            if ((not self.decode_mod(insn, insn.Op2, m2, size_b, insndat)) or
                (not self.decode_ext(insn, insn.Op3))):
                return False
        elif subfmt == 3: # VIId (VIIc for BCD instructions)
            if ((not self.decode_mod(insn, insn.Op2, m2, size_b, insndat)) or
                (not self.decode_imm8(insn, insn.Op3))):
                return False

        insn.auxpref |= self.generate_postfix(insn, insndat, param)
        insn.itype = getattr(self, 'itype_' + insndat['name'])
        return True

    # ----------------------------------------------------------------------
    # does operand match tuple m? (type, value)
    def match_op(self, op, m):
        if m == None:
            return True
        if op.type != m[0]:
            return False
        if op.type == o_imm:
            return op.value == m[1]
        elif op.type in [o_reg, o_phrase]:
            return op.reg == m[1]
        else:
            return false

    # ----------------------------------------------------------------------
    def notify_ana(self, insn):
        """
        Decodes an instruction into 'insn'.
        Returns: insn.size (=the size of the decoded instruction) or zero
        """
        op = insn.get_next_byte()
        subop = op

        # Unhandled opcode
        if op not in self.opcode_table['BASE']:
            if VerboseFailure:
                raise Exception("Unhandled opcode %X (@%X)" % (op, insn.ea))
            return 0

        insndat = self.opcode_table['BASE'][op]
        param = self.opcode_param['BASE'][subop]

        # Prefix opcode
        if insndat == 'PREFIX':
            subop = insn.get_next_byte()
            table = self.opcode_param['BASE'][op]
            if table not in self.opcode_table:
                raise Exception('Prefix table %s not found' % (table))
                return 0
            if subop not in self.opcode_table[table]:
                if VerboseFailure:
                    raise Exception("Unhandled opcode %X:%X (@%X, %s)" % (op, subop, insn.ea, table))
                return 0
            insndat = self.opcode_table[table][subop]
            param = self.opcode_param[table][subop]

        passed = False
        if insndat['format'] == InsnFmt_I:
            passed = self.decode_fmt_I(insn, op, subop, insndat, param)
        elif insndat['format'] == InsnFmt_II:
            passed = self.decode_fmt_II(insn, op, subop, insndat, param)
        elif insndat['format'] == InsnFmt_I_II:
            passed = self.decode_fmt_I_II(insn, op, subop, insndat, param)
        elif insndat['format'] == InsnFmt_III:
            passed = self.decode_fmt_III(insn, op, subop, insndat, param)
        elif insndat['format'] == InsnFmt_IV:
            passed = self.decode_fmt_IV(insn, op, subop, insndat, param)
        elif insndat['format'] == InsnFmt_V:
            passed = self.decode_fmt_V(insn, op, subop, insndat, param)
        elif insndat['format'] == InsnFmt_VI:
            passed = self.decode_fmt_VI(insn, op, subop, insndat, param)
        elif insndat['format'] == InsnFmt_VIIa:
            passed = self.decode_fmt_VII(insn, op, subop, insndat, param, 0)
        elif insndat['format'] == InsnFmt_VIIb:
            passed = self.decode_fmt_VII(insn, op, subop, insndat, param, 1)
        elif insndat['format'] == InsnFmt_VIIc:
            passed = self.decode_fmt_VII(insn, op, subop, insndat, param, 2)
        elif insndat['format'] == InsnFmt_VIId:
            passed = self.decode_fmt_VII(insn, op, subop, insndat, param, 3)

        if not passed:
            if VerboseFailure:
                raise Exception("Decode %X failed" % (op))
            insn.itype = self.itype_null

        # Return decoded instruction size or zero
        return insn.size if insn.itype != self.itype_null else 0

    # ----------------------------------------------------------------------
    def init_instructions(self):
        Instructions = []
        i = 0
        for x in self.instruc:
            if x['name'] != '':
                setattr(self, 'itype_' + x['name'], i)
            else:
                setattr(self, 'itype_null', i)
            i += 1

        self.opcode_table = {}
        self.opcode_param = {}
        for pfx in ('BASE', 'FPU', 'CVT', 'BCD', 'DBcc', 'DBNcc'):
            self.opcode_table[pfx] = {}
            self.opcode_param[pfx] = {}
        for x in self.decoder_table:
            for l in range(256):
                parameters = {}
                v = l
                passed = True
                for c in x['valuemask']:
                    if (c == '0') and ((v & 0x80) != 0x00):
                        passed = False
                        break
                    if (c == '1') and ((v & 0x80) != 0x80):
                        passed = False
                        break
                    if (c == 's') or (c == 'S') or (c == 'm'):
                        if c not in parameters:
                            parameters[c] = 0
                        else:
                            parameters[c] <<= 1
                        parameters[c] |= (v >> 7) & 1
                    v <<= 1

                # Only allow DWORD for instructions that support it
                if 's' in parameters and parameters['s'] == DataLen_Dword:
                    if not ('feature' in x and ('DWORD_OP1' in x['feature'] or 'DWORD_MUST_BOTH' in x['feature'])):
                        passed = False
                if 'S' in parameters and parameters['S'] == DataLen_Dword:
                    if not ('feature' in x and ('DWORD_OP2' in x['feature'] or 'DWORD_MUST_BOTH' in x['feature'])):
                        passed = False

                # Only allow DWORD if operands are the same size
                if 'feature' in x and 'DWORD_MUST_BOTH' in x['feature']:
                    if (('S' in parameters and 's' in parameters) and
                        (parameters['s'] == DataLen_Dword or parameters['S'] == DataLen_Dword)):
                        if parameters['s'] != parameters['S']:
                            passed = False

                if 'size' in x:
                    if not isinstance(x['size'], (str, unicode)):
                        # size array
                        parameters['s'] = sizename_datalen_mapping[x['size'][0]]
                        parameters['S'] = sizename_datalen_mapping[x['size'][1]]
                    else:
                        # single size
                        parameters['s'] = sizename_datalen_mapping[x['size']]

                if passed:
                    if 'prefix' in x:
                        self.opcode_table[x['prefix']][l] = x
                        self.opcode_param[x['prefix']][l] = parameters
                    else:
                        self.opcode_table['BASE'][l] = x
                        self.opcode_param['BASE'][l] = parameters

        # Install prefixes
        self.opcode_table['BASE'][0x59] = 'PREFIX'
        self.opcode_param['BASE'][0x59] = 'BCD'
        self.opcode_table['BASE'][0x5C] = 'PREFIX'
        self.opcode_param['BASE'][0x5C] = 'FPU'
        self.opcode_table['BASE'][0x5E] = 'PREFIX'
        self.opcode_param['BASE'][0x5E] = 'FPU'
        self.opcode_table['BASE'][0x5F] = 'PREFIX'
        self.opcode_param['BASE'][0x5F] = 'CVT'
        self.opcode_table['BASE'][0xC6] = 'PREFIX'
        self.opcode_param['BASE'][0xC6] = 'DBcc'
        self.opcode_table['BASE'][0xC7] = 'PREFIX'
        self.opcode_param['BASE'][0xC7] = 'DBNcc'

        # icode of the last instruction + 1
        self.instruc_end = len(self.instruc) + 1

        # Icode of return instruction. It is ok to give any of possible return
        # instructions
        self.icode_return = self.itype_ret

    # ----------------------------------------------------------------------
    def init_registers(self):
        """This function parses the register table and creates corresponding ireg_XXX constants"""

        # Registers definition
        self.reg_names = [
            # General purpose registers
            "R0",
            "R1",
            "R2",
            "R3",
            "R4",
            "R5",
            "R6",
            "R7",
            "R8",
            "R9",
            "R10",
            "R11",
            "R12",
            "R13",
            "R14",
            "R15",
            "R16",
            "R17",
            "R18",
            "R19",
            "R20",
            "R21",
            "R22",
            "R23",
            "R24",
            "R25",
            "R26",
            "R27",
            "R28",
            "AP", # R29
            "FP", # R30
            "SP", # R31
            # Fake PC register
            "PC",
            # Fake segment registers
            "CS",
            "DS"
        ]

        # Create the ireg_XXXX constants
        for i in xrange(len(self.reg_names)):
            setattr(self, 'ireg_' + self.reg_names[i], i)

        # Segment register information (use virtual CS and DS registers if your
        # processor doesn't have segment registers):
        self.reg_first_sreg = self.ireg_CS
        self.reg_last_sreg  = self.ireg_DS

        # number of CS register
        self.reg_code_sreg = self.ireg_CS

        # number of DS register
        self.reg_data_sreg = self.ireg_DS

    # ----------------------------------------------------------------------
    def __init__(self):
        processor_t.__init__(self)
        self.init_instructions()
        self.init_registers()
        # TODO: create privileged register enum

# ----------------------------------------------------------------------
# Every processor module script must provide this function.
# It should return a new instance of a class derived from processor_t
def PROCESSOR_ENTRY():
    return necv60_processor_t()
